package fr.univlorraine.tools.vaadin.theme;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Writer;
import java.lang.reflect.Method;
import java.net.URL;
import java.net.URLClassLoader;
import java.nio.file.FileSystemNotFoundException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Collections;
import java.util.Map;
import java.util.Properties;
import java.util.stream.Stream;

import org.springframework.core.env.MutablePropertySources;
import org.springframework.core.env.PropertiesPropertySource;
import org.springframework.core.env.PropertyResolver;
import org.springframework.core.env.PropertySources;
import org.springframework.core.env.PropertySourcesPropertyResolver;
import org.springframework.core.io.Resource;
import org.springframework.core.io.support.PathMatchingResourcePatternResolver;

import com.vaadin.sass.internal.ScssStylesheet;

import lombok.Getter;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;

/**
 * Permet de personnaliser un theme Vaadin en générant les fichiers .scss à partir des .tscss, et en le compilant.
 * Les tokens présents dans les fichiers .tscss sont remplacés selon les propertySources passées.
 * Si com.yahoo.platform.yui.compressor.CssCompressor est disponible, il est possible de minifier le fichier styles.css généré.
 * @author Adrien Colson
 */
@Slf4j
public class ThemeCustomizer {

	/**
	 * Erreur lors de la personnalisation du theme.
	 */
	@SuppressWarnings("serial")
	public class ThemeCustomizerException extends Exception {
		/**
		 * @param message message
		 */
		public ThemeCustomizerException(final String message) {
			super(message);
		}

		/**
		 * @param message message
		 * @param cause cause
		 */
		public ThemeCustomizerException(final String message, final Throwable cause) {
			super(message, cause);
		}
	}

	/** Dossier des themes Vaadin. */
	private static final String THEMES_FOLDER = "VAADIN/themes/";
	/** Suffixe theme personnalisé. */
	private static final String FILTERABLE_EXTENSION = ".tscss";
	/** Dossier des themes Vaadin. */
	private static final String THEME_NAME_TOKEN = "THEME_NAME";
	/** Classe effectuant la minification. */
	private static final String MINIFY_CLASS = "com.yahoo.platform.yui.compressor.CssCompressor";

	private final PropertySources globalPropertySources;

	@Getter @Setter
	private boolean minifyStylesCss = isMinifyClassAvailable();

	/**
	 * Constructeur.
	 * @param propertySources propertySources
	 */
	public ThemeCustomizer(final PropertySources propertySources) {
		globalPropertySources = propertySources;
	}

	/**
	 * @return true s'il est possible de minifier.
	 */
	private boolean isMinifyClassAvailable() {
		try {
			Class.forName(MINIFY_CLASS);
		} catch (final ClassNotFoundException e) {
			return false;
		}
		return true;
	}

	/**
	 * Personnalise un theme Vaadin.
	 * @param themeName nom du theme à personnaliser
	 * @param customThemeName nom du theme personnalisé
	 * @throws ThemeCustomizerException erreur lors de la personnalisation du theme
	 */
	public void customizeTheme(final String themeName, final String customThemeName) throws ThemeCustomizerException {
		customizeTheme(themeName, customThemeName, Collections.emptyMap());
	}

	/**
	 * Personnalise un theme Vaadin.
	 * @param themeName nom du theme à personnaliser
	 * @param customThemeName nom du theme personnalisé
	 * @param filesToCopy fichiers supplémentaires à copier
	 * @throws ThemeCustomizerException erreur lors de la personnalisation du theme
	 */
	public void customizeTheme(final String themeName, final String customThemeName, final Map<String, String> filesToCopy) throws ThemeCustomizerException {
		/* Crée le dossier temporaire */
		Path tempFolderPath;
		try {
			tempFolderPath = Files.createTempDirectory(customThemeName);
		} catch (final IOException e) {
			throw new ThemeCustomizerException(String.format("Impossible de créer un dossier temporaire pour la personnalisation du theme %s.", themeName), e);
		}
		/* Supprime le dossier à la fermeture de l'application */
		Runtime.getRuntime().addShutdownHook(new Thread(() -> {
			try {
				Files.walk(tempFolderPath)
					.map(Path::toFile)
					.sorted((o1, o2) -> -o1.compareTo(o2))
					.forEach(File::delete);
			} catch (final IOException e) {
				log.warn("Impossible de supprimer le dossier temporaire {}", tempFolderPath, e);
			}
		}));

		/* Copie les fichiers supplémentaires */
		filesToCopy.forEach((source, dest) -> {
			log.info("Copie {} vers {}...", source, dest);
			try {
				final Path sourceFile = Paths.get(source);
				final Path destFile = tempFolderPath.resolve(THEMES_FOLDER + customThemeName).resolve(dest);
				Files.createDirectories(destFile.getParent());
				Files.copy(sourceFile, destFile);
			} catch (final Exception e) {
				log.error("Impossible de copier {} dans le theme personnalisé {}.", source, themeName, e);
			}
		});

		/* Filtre les fichiers .tscss */
		try {
			copyAndFilterFiles(themeName, customThemeName, tempFolderPath);
		} catch (final Exception e) {
			throw new ThemeCustomizerException(String.format("Une erreur s'est produite lors de la personnalisation du theme %s.", themeName), e);
		}

		/* Ajoute le dossier au classpath */
		try {
			addPath(tempFolderPath.toUri().toURL());
		} catch (final Exception e) {
			throw new ThemeCustomizerException(String.format("Impossible d'ajouter au classpath le dossier temporaire du theme personnalisé %s.", themeName), e);
		}

		/* Compile styles.css */
		final ScssStylesheet scssStylesheet;
		final String scssFile = THEMES_FOLDER + customThemeName + "/styles.scss";
		try {
			scssStylesheet = ScssStylesheet.get(scssFile);

			if (scssStylesheet == null) {
				throw new ThemeCustomizerException(String.format("Impossible d'accéder à %s.", scssFile));
			}

			scssStylesheet.compile();
		} catch (final Exception e) {
			throw new ThemeCustomizerException(String.format("Une erreur s'est produite lors de la compilation du theme personnalisé %s.", themeName), e);
		}

		/* Ecrit styles.css */
		final Path cssPath = tempFolderPath.resolve(THEMES_FOLDER + customThemeName).resolve("styles.css");
		log.info("Génère {}...", cssPath);
		try {
			final Writer writer = Files.newBufferedWriter(cssPath);
			if (minifyStylesCss) {
				final boolean minify = isMinifyClassAvailable();
				if (!minify) {
					log.warn("Impossible de minifier styles.css, {} non disponible.", MINIFY_CLASS);
				}
				scssStylesheet.write(writer, minify);
			} else {
				scssStylesheet.write(writer);
			}
			writer.close();
		} catch (final IOException e) {
			throw new ThemeCustomizerException(String.format("Impossible d'écrire le .css compilé du theme personnalisé %s.", themeName), e);
		}
	}

	/**
	 * Génère les fichiers .scss à partir des .tscss, en remplacant les tokens selon les propriétés Spring.
	 * @param sourceThemeName nom du theme source
	 * @param targetThemeName nom du theme cible
	 * @param tempPath dossier temporaire
	 * @throws FileSystemNotFoundException les ressources ne sont pas sur le système de fichier (servies depuis un .war par exemple)
	 * @throws IllegalArgumentException toutes les propriétés nécessaires ne sont pas renseignées
	 * @throws IOException IOException erreur IO
	 */
	private void copyAndFilterFiles(final String sourceThemeName, final String targetThemeName, final Path tempPath) throws FileSystemNotFoundException, IllegalArgumentException, IOException {
		/* Liste les fichiers à filtrer */
		final PathMatchingResourcePatternResolver resolver = new PathMatchingResourcePatternResolver();
		final String searchPattern = THEMES_FOLDER + sourceThemeName + "/**/*";
		final Resource[] resourcesToFilter = resolver.getResources(searchPattern);

		/* Initialise les propriétés du theme personnalisé */
		final Properties themeProperties = new Properties();
		themeProperties.setProperty(THEME_NAME_TOKEN, targetThemeName);
		final PropertiesPropertySource themePropertySource = new PropertiesPropertySource("themePropertySource", themeProperties);
		/* Initialise le propertyResolver */
		final MutablePropertySources themePropertySources = new MutablePropertySources(globalPropertySources);
		themePropertySources.addFirst(themePropertySource);
		final PropertyResolver propertyResolver = new PropertySourcesPropertyResolver(themePropertySources);

		/* Filtre chaque fichier */
		final int sourceThemePathLength = resolver.getResource(THEMES_FOLDER + sourceThemeName + '/').getURL().getPath().length();
		final Path targetThemePath = tempPath.resolve(THEMES_FOLDER + targetThemeName);
		for (final Resource resourceToFilter : resourcesToFilter) {
			final String resourceClassPath = resourceToFilter.getURL().getPath().substring(sourceThemePathLength);
			final boolean filterFile = resourceClassPath.endsWith(FILTERABLE_EXTENSION);
			final Path targetPath = targetThemePath.resolve(filterFile ? resourceClassPath.replaceAll('\\' + FILTERABLE_EXTENSION + '$', ".scss") : resourceClassPath);

			Files.createDirectories(targetPath.getParent());
			if (filterFile) {
				/* Filtre le fichier avant de le recopier */
				log.info("Génère {}...", targetPath);
				final Stream<String> filteredLinesStream = new BufferedReader(new InputStreamReader(resourceToFilter.getInputStream()))
					.lines()
					.map(propertyResolver::resolveRequiredPlaceholders);
				Files.write(targetPath, (Iterable<String>) filteredLinesStream::iterator);
			} else if (!Files.exists(targetPath)) {
				/* Recopie le fichier s'il n'existe pas déjà */
				log.info("Copie {}...", targetPath);
				Files.copy(resourceToFilter.getInputStream(), targetPath);
			}
		}
	}

	/**
	 * Ajoute le dossier au classpath.
	 * @param url url du dossier
	 * @throws Exception Exception
	 */
	public void addPath(final URL url) throws Exception {
		final URLClassLoader urlClassLoader = (URLClassLoader) ClassLoader.getSystemClassLoader();
		final Method method = URLClassLoader.class.getDeclaredMethod("addURL", new Class[]{URL.class});
		method.setAccessible(true);
		method.invoke(urlClassLoader, new Object[]{url});
	}

}
