package fr.univlorraine.tools.vaadin.theme;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Writer;
import java.lang.reflect.Method;
import java.net.URL;
import java.net.URLClassLoader;
import java.nio.file.FileSystemNotFoundException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Collections;
import java.util.Map;
import java.util.stream.Stream;

import org.springframework.core.env.Environment;
import org.springframework.core.io.Resource;
import org.springframework.core.io.support.PathMatchingResourcePatternResolver;

import com.vaadin.sass.internal.ScssStylesheet;

import lombok.extern.slf4j.Slf4j;

/**
 * Permet de personnaliser un theme Vaadin en générant les fichiers .scss à partir des .tscss, et en le recompilant.
 * Les tokens présents dans les fichiers .tscss sont remplacés par les propriétés Spring.
 * Attention, si les fichiers du theme ne sont pas présents sur le disque (servis depuis un .war par exemple), cela ne peut fonctionner.
 * @author Adrien Colson
 */
@Slf4j
public class ThemeCustomizer {

	/** Dossier des themes Vaadin. */
	private static final String THEMES_FOLDER = "VAADIN/themes/";

	private final Environment env;

	/**
	 * Constructeur.
	 * @param environment environment Spring
	 */
	public ThemeCustomizer(final Environment environment) {
		env = environment;
	}

	/**
	 * Personnalise un theme Vaadin.
	 * @param themeName nom du theme à personnaliser
	 */
	public void customizeTheme(final String themeName) {
		customizeTheme(themeName, Collections.emptyMap());
	}

	/**
	 * Personnalise un theme Vaadin.
	 * @param themeName nom du theme à personnaliser
	 * @param filesToCopy fichiers supplémentaires à copier
	 */
	public void customizeTheme(final String themeName, final Map<String, String> filesToCopy) {
		/* Crée le dossier temporaire */
		Path tempFolderPath;
		try {
			tempFolderPath = Files.createTempDirectory("actul-custom-theme");
		} catch (final IOException e) {
			log.error("Impossible de créer un dossier temporaire pour la personnalisation du theme {}.", themeName, e);
			return;
		}
		/* Supprime le dossier à la fermeture de l'application */
		Runtime.getRuntime().addShutdownHook(new Thread() {
			@Override
			public void run() {
				try {
					Files.walk(tempFolderPath)
						.map(Path::toFile)
						.sorted((o1, o2) -> -o1.compareTo(o2))
						.forEach(File::delete);
				} catch (final IOException e) {
					e.printStackTrace();
				}
			}
		});

		final String themeFolder = THEMES_FOLDER + themeName;

		/* Copie les fichiers supplémentaires */
		filesToCopy.forEach((source, dest) -> {
			try {
				final Path sourceFile = Paths.get(source);
				final Path destFile = tempFolderPath.resolve(themeFolder).resolve(dest);
				log.info("Copie {}...", sourceFile);
				Files.createDirectories(destFile.getParent());
				Files.copy(sourceFile, destFile);
			} catch (final Exception e) {
				log.error("Impossible de copier {} dans le theme personnalisé {}.", source, themeName, e);
			}
		});

		/* Filtre les fichiers .tscss */
		try {
			filterFiles(themeFolder, tempFolderPath);
		} catch (final IOException e) {
			log.error("Une erreur s'est produite lors de la personnalisation du theme {}.", themeName, e);
			return;
		}

		/* Ajoute le dossier au classpath*/
		try {
			addPath(tempFolderPath.toUri().toURL());
		} catch (final Exception e) {
			log.error("Impossible d'ajouter au classpath le dossier temporaire du theme personnalisé {}.", themeName, e);
			return;
		}

		/* Compile styles.css */
		final ScssStylesheet scssStylesheet;
		final String scssFile = themeFolder + "/styles.scss";
		try {
			scssStylesheet = ScssStylesheet.get(scssFile);

			if (scssStylesheet == null) {
				log.error("Impossible d'accéder à {}.", scssFile);
				return;
			}

			scssStylesheet.compile();
		} catch (final Exception e) {
			log.error("Une erreur s'est produite lors de la compilation du theme personnalisé {}.", themeName, e);
			return;
		}

		/* Ecrit styles.css */
		final Path cssPath = tempFolderPath.resolve(themeFolder).resolve("styles.css");
		log.info("Génère {}...", cssPath);
		try {
			final Writer writer = Files.newBufferedWriter(cssPath);
			scssStylesheet.write(writer);
			writer.close();
		} catch (final IOException e) {
			log.error("Impossible d'écrire le .css compilé du theme personnalisé {}.", themeName, e);
			return;
		}
	}

	/**
	 * Génère les fichiers .scss à partir des .tscss, en remplacant les tokens selon les propriétés Spring.
	 * @param resourcesPath chemin des fichiers
	 * @param tempThemeFolderPath dossier de theme temporaire
	 * @throws FileSystemNotFoundException les ressources ne sont pas sur le système de fichier (servies depuis un .war par exemple)
	 * @throws IllegalArgumentException toutes les propriétés nécessaires ne sont pas renseignées
	 * @throws IOException IOException erreur IO
	 */
	private void filterFiles(final String resourcesPath, final Path tempThemeFolderPath) throws FileSystemNotFoundException, IllegalArgumentException, IOException {
		/* Liste les fichiers à filtrer */
		final PathMatchingResourcePatternResolver resolver = new PathMatchingResourcePatternResolver();
		final String searchPattern = resourcesPath + "/**/*.tscss";
		final Resource[] resourcesToFilter = resolver.getResources(searchPattern);

		/* Filtre chaque fichier */
		final String rootPath = resolver.getResource("/").getURL().getPath();
		for (final Resource resourceToFilter : resourcesToFilter) {
			final String resourceClassPath = resourceToFilter.getURL().getPath().substring(rootPath.length());
			final Path targetPath = tempThemeFolderPath.resolve(resourceClassPath.replaceAll("\\.tscss$", ".scss"));

			final Stream<String> filteredLinesStream = new BufferedReader(new InputStreamReader(resourceToFilter.getInputStream()))
				.lines()
				.map(env::resolveRequiredPlaceholders);

			log.info("Génère {}...", targetPath);
			Files.createDirectories(targetPath.getParent());
			Files.write(targetPath, (Iterable<String>) filteredLinesStream::iterator);
		}
	}

	/**
	 * Ajoute le dossier au classpath.
	 * @param url url du dossier
	 * @throws Exception Exception
	 */
	public void addPath(final URL url) throws Exception {
		final URLClassLoader urlClassLoader = (URLClassLoader) ClassLoader.getSystemClassLoader();
		final Method method = URLClassLoader.class.getDeclaredMethod("addURL", new Class[]{URL.class});
		method.setAccessible(true);
		method.invoke(urlClassLoader, new Object[]{url});
	}

}
