package fr.univlorraine.tools.test.utils;

import java.io.File;
import java.io.IOException;
import java.util.function.Supplier;

import org.junit.rules.TestWatcher;
import org.junit.runner.Description;
import org.openqa.selenium.OutputType;
import org.openqa.selenium.TakesScreenshot;
import org.springframework.util.FileCopyUtils;

import lombok.Getter;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;

/**
 * Screenshot on test fail.
 * @see <a href="https://alexecollins.com/taking-a-screeshot-with-selenium-web-driver">https://alexecollins.com/taking-a-screeshot-with-selenium-web-driver</a>
 */
@Slf4j
public class ScreenshotRule extends TestWatcher {

	/** classname_token. */
	public static final String CLASSNAME_TOKEN = "${classname}";
	/** methodname_token. */
	public static final String METHODNAME_TOKEN = "${methodname}";
	/** dot_extension. */
	public static final String DOT_EXTENSION = ".png";

	@Getter @Setter
	private Supplier<TakesScreenshot> takesScreenshotSupplier;
	@Getter
	private String screenshotsFolder;
	@Getter
	private String screenshotsFilename;

	/**
	 * @param newScreenshotsFolder new screenshotsFolder, can include tokens
	 */
	private void setScreenshotsFolder(final String newScreenshotsFolder) {
		screenshotsFolder = newScreenshotsFolder == null ? "." : newScreenshotsFolder;
	}

	/**
	 * @param newScreenshotsFilename new screenshotsFolder, can include tokens
	 */
	private void setScreenshotsFilename(final String newScreenshotsFilename) {
		screenshotsFilename = newScreenshotsFilename == null ? "failed_" + CLASSNAME_TOKEN + "." + METHODNAME_TOKEN + DOT_EXTENSION : newScreenshotsFilename;
	}

	/**
	 * @param newTakesScreenshotSupplier web driver supplier
	 * @param newScreenshotsFolder screenshots folder, can include tokens
	 * @param newScreenshotsFilename screenshots filename, can include tokens
	 */
	public ScreenshotRule(final Supplier<TakesScreenshot> newTakesScreenshotSupplier, final String newScreenshotsFolder, final String newScreenshotsFilename) {
		super();
		takesScreenshotSupplier = newTakesScreenshotSupplier;
		setScreenshotsFolder(newScreenshotsFolder);
		setScreenshotsFilename(newScreenshotsFilename);
	}

	/**
	 * @param newTakesScreenshotSupplier web driver supplier
	 * @param newScreenshotsFolder screenshots folder, can include tokens
	 */
	public ScreenshotRule(final Supplier<TakesScreenshot> newTakesScreenshotSupplier, final String newScreenshotsFolder) {
		this(newTakesScreenshotSupplier, newScreenshotsFolder, null);
	}

	/**
	 * @param newTakesScreenshotSupplier web driver supplier
	 */
	public ScreenshotRule(final Supplier<TakesScreenshot> newTakesScreenshotSupplier) {
		this(newTakesScreenshotSupplier, null);
	}

	/**
	 * @param takesScreenshot web driver
	 * @param newScreenshotsFolder screenshots folder, can include tokens
	 * @param newScreenshotsFilename screenshots filename, can include tokens
	 */
	public ScreenshotRule(final TakesScreenshot takesScreenshot, final String newScreenshotsFolder, final String newScreenshotsFilename) {
		this(() -> takesScreenshot, newScreenshotsFolder, newScreenshotsFilename);
	}

	/**
	 * @param takesScreenshot web driver
	 * @param newScreenshotsFolder screenshots folder, can include tokens
	 */
	public ScreenshotRule(final TakesScreenshot takesScreenshot, final String newScreenshotsFolder) {
		this(takesScreenshot, newScreenshotsFolder, null);
	}

	/**
	 * @param takesScreenshot web driver
	 */
	public ScreenshotRule(final TakesScreenshot takesScreenshot) {
		this(takesScreenshot, null);
	}

	/**
	 * @see org.junit.rules.TestWatcher#failed(java.lang.Throwable, org.junit.runner.Description)
	 */
	@Override
	protected void failed(final Throwable throwable, final Description description) {
		final String folder = parseString(screenshotsFolder, description);
		final String filename = parseString(screenshotsFilename, description);
		try {
			final boolean dirCreated = new File(folder).mkdirs();
			if (dirCreated) {
				log.info("Screenshots directory created.");
			}
			final byte[] screenshot = takesScreenshotSupplier.get().getScreenshotAs(OutputType.BYTES);
			final File outputfile = new File(folder + "/" + filename);
			FileCopyUtils.copy(screenshot, outputfile);
		} catch (final IOException e) {
			log.error("Failed to take screenshot.", e);
		}
	}

	/**
	 * @param toParse string to parse
	 * @param description test description
	 * @return string with tokens replaced
	 */
	private String parseString(final String toParse, final Description description) {
		return toParse.replace(CLASSNAME_TOKEN, description.getClassName())
			.replace(METHODNAME_TOKEN, description.getMethodName());
	}

}
