package fr.univlorraine.tools.vaadin.spring.security;

import org.springframework.security.core.context.SecurityContext;
import org.springframework.security.core.context.SecurityContextHolderStrategy;
import org.springframework.security.core.context.SecurityContextImpl;
import org.springframework.util.Assert;

import com.vaadin.server.VaadinSession;

/**
 * A custom {@link SecurityContextHolderStrategy} that stores the {@link SecurityContext} in the Vaadin Session and in an InheritableThreadLocal CONTEXT_HOLDER.
 * @see <a href="https://github.com/peholmst/SpringSecurityDemo/blob/master/hybrid-security/src/main/java/org/vaadin/peholmst/samples/springsecurity/hybrid/VaadinSessionSecurityContextHolderStrategy.java">SpringSecurityDemo/.../VaadinSessionSecurityContextHolderStrategy.java</a>
 * @author Adrien Colson
 *
 * @see java.lang.ThreadLocal
 */
public class VaadinSecurityContextHolderStrategy implements SecurityContextHolderStrategy {

	/** startup_context_holder. */
	private static final ThreadLocal<SecurityContext> STARTUP_CONTEXT_HOLDER = new ThreadLocal<SecurityContext>();
	/** inheritable_context_holder. */
	private static final ThreadLocal<SecurityContext> INHERITABLE_CONTEXT_HOLDER = new InheritableThreadLocal<SecurityContext>();

	/**
	 * @see org.springframework.security.core.context.SecurityContextHolderStrategy#clearContext()
	 */
	@Override
	public void clearContext() {
		STARTUP_CONTEXT_HOLDER.remove();
		INHERITABLE_CONTEXT_HOLDER.remove();
	}

	/**
	 * @see org.springframework.security.core.context.SecurityContextHolderStrategy#getContext()
	 */
	@Override
	public SecurityContext getContext() {
		SecurityContext ctx;
		final VaadinSession vaadinSession = VaadinSession.getCurrent();
		final SecurityContext startupSecurityContext = STARTUP_CONTEXT_HOLDER.get();
		if (vaadinSession == null) {
			if (startupSecurityContext == null) {
				ctx = INHERITABLE_CONTEXT_HOLDER.get();
			} else {
				ctx = startupSecurityContext;
			}
		} else {
			if (startupSecurityContext == null) {
				ctx = vaadinSession.getAttribute(SecurityContext.class);
			} else {
				STARTUP_CONTEXT_HOLDER.remove();
				vaadinSession.setAttribute(SecurityContext.class, startupSecurityContext);
				ctx = startupSecurityContext;
			}
		}

		if (ctx == null) {
			ctx = createEmptyContext();
			setContext(ctx);
		}

		return ctx;
	}

	/**
	 * @see org.springframework.security.core.context.SecurityContextHolderStrategy#setContext(org.springframework.security.core.context.SecurityContext)
	 */
	@Override
	public void setContext(final SecurityContext context) {
		Assert.notNull(context, "Only non-null SecurityContext instances are permitted");

		INHERITABLE_CONTEXT_HOLDER.set(context);

		final VaadinSession vaadinSession = VaadinSession.getCurrent();
		if (vaadinSession == null) {
			STARTUP_CONTEXT_HOLDER.set(context);
		} else {
			STARTUP_CONTEXT_HOLDER.remove();
			vaadinSession.setAttribute(SecurityContext.class, context);
		}
	}

	/**
	 * @see org.springframework.security.core.context.SecurityContextHolderStrategy#createEmptyContext()
	 */
	@Override
	public SecurityContext createEmptyContext() {
		return new SecurityContextImpl();
	}

}
