package fr.univlorraine.tools.test.utils;

import java.io.File;
import java.io.IOException;
import java.util.function.Supplier;

import org.junit.rules.TestWatcher;
import org.junit.runner.Description;
import org.openqa.selenium.OutputType;
import org.openqa.selenium.TakesScreenshot;
import org.springframework.util.FileCopyUtils;

import lombok.Getter;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;

/**
 * Screenshot on test fail.
 * @see <a href="https://alexecollins.com/taking-a-screeshot-with-selenium-web-driver">https://alexecollins.com/taking-a-screeshot-with-selenium-web-driver</a>
 */
@Slf4j
public class ScreenshotRule extends TestWatcher {

	@Getter @Setter
	private Supplier<TakesScreenshot> takesScreenshotSupplier;
	@Getter
	private String screenshotsFolder;

	/**
	 * @param newScreenshotsFolder new screenshotsFolder
	 */
	private void setScreenshotsFolder(final String newScreenshotsFolder) {
		screenshotsFolder = newScreenshotsFolder == null ? "" : newScreenshotsFolder;
	}

	/**
	 * @param newTakesScreenshotSupplier web driver supplier
	 * @param newScreenshotsFolder screenshots folder
	 */
	public ScreenshotRule(final Supplier<TakesScreenshot> newTakesScreenshotSupplier, final String newScreenshotsFolder) {
		super();
		takesScreenshotSupplier = newTakesScreenshotSupplier;
		setScreenshotsFolder(newScreenshotsFolder);
	}

	/**
	 * @param takesScreenshot web driver
	 * @param newScreenshotsFolder screenshots folder
	 */
	public ScreenshotRule(final TakesScreenshot takesScreenshot, final String newScreenshotsFolder) {
		this(() -> takesScreenshot, newScreenshotsFolder);
	}

	/**
	 * @param newTakesScreenshotSupplier web driver supplier
	 */
	public ScreenshotRule(final Supplier<TakesScreenshot> newTakesScreenshotSupplier) {
		this(newTakesScreenshotSupplier, null);
	}

	/**
	 * @param takesScreenshot web driver
	 */
	public ScreenshotRule(final TakesScreenshot takesScreenshot) {
		this(takesScreenshot, null);
	}

	/**
	 * @see org.junit.rules.TestWatcher#failed(java.lang.Throwable, org.junit.runner.Description)
	 */
	@Override
	protected void failed(final Throwable throwable, final Description description) {
		try {
			final boolean dirCreated = new File(screenshotsFolder).mkdirs();
			if (dirCreated) {
				log.info("Screenshots directory created.");
			}
			final byte[] screenshot = takesScreenshotSupplier.get().getScreenshotAs(OutputType.BYTES);
			final File outputfile = new File(screenshotsFolder + "/failed_" + description.getClassName() + "." + description.getMethodName() + ".png");
			FileCopyUtils.copy(screenshot, outputfile);
		} catch (final IOException e) {
			log.error("Failed to take screenshot.", e);
		}
	}

}
