package fr.univlorraine.tools.test.utils;

import java.io.IOException;
import java.util.Collection;

import javax.servlet.FilterChain;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.springframework.boot.test.context.TestComponent;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.Order;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.authority.AuthorityUtils;
import org.springframework.security.core.context.SecurityContext;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.core.userdetails.User;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.security.test.web.support.WebTestUtils;
import org.springframework.security.web.context.HttpRequestResponseHolder;
import org.springframework.security.web.context.SecurityContextRepository;
import org.springframework.web.filter.GenericFilterBean;

import lombok.Getter;
import lombok.Setter;

/**
 * Permet de simuler l'authentification d'un utilisateur à des fins de tests.
 * @see <a href="http://stackoverflow.com/questions/30034761/spring-security-withmockuser-for-selenium-test">http://stackoverflow.com/questions/30034761/spring-security-withmockuser-for-selenium-test</a>
 */
@TestComponent
@Order(Ordered.HIGHEST_PRECEDENCE)
public class MockUserFilter extends GenericFilterBean {

	@Getter @Setter
	private SecurityContext securityContext;

	/**
	 * @see javax.servlet.Filter#doFilter(javax.servlet.ServletRequest, javax.servlet.ServletResponse, javax.servlet.FilterChain)
	 */
	@Override
	public void doFilter(final ServletRequest request, final ServletResponse response, final FilterChain chain) throws IOException, ServletException {
		HttpServletRequest servletRequest = (HttpServletRequest) request;
		HttpServletResponse servletResponse = (HttpServletResponse) response;

		if (securityContext != null) {
			final SecurityContextRepository securityContextRepository = WebTestUtils.getSecurityContextRepository(servletRequest);
			final HttpRequestResponseHolder requestResponseHolder = new HttpRequestResponseHolder(servletRequest, servletResponse);
			securityContextRepository.loadContext(requestResponseHolder);
			servletRequest = requestResponseHolder.getRequest();
			servletResponse = requestResponseHolder.getResponse();
			securityContextRepository.saveContext(securityContext, servletRequest, servletResponse);
			securityContext = null;
		}

		chain.doFilter(request, response);
	}

	/**
	 * Authentifie la prochaine requête.
	 */
	public void authenticateNextRequest() {
		authenticateNextRequestAs("user");
	}

	/**
	 * Authentifie la prochaine requête avec un nom d'utilisateur et des droits donnés.
	 * @param username username
	 * @param roles authorities list
	 */
	public void authenticateNextRequestAs(final String username, final String... roles) {
		final Collection<GrantedAuthority> authorities = AuthorityUtils.createAuthorityList(roles);
		final UserDetails principal = new User(username, "password", authorities);
		final Authentication authentication = new UsernamePasswordAuthenticationToken(principal, principal.getPassword(), principal.getAuthorities());

		securityContext = SecurityContextHolder.createEmptyContext();
		securityContext.setAuthentication(authentication);
		SecurityContextHolder.setContext(securityContext);
	}

}
