package fr.univlorraine.tools.reactive;

import java.time.Duration;
import java.util.List;

import reactor.core.publisher.Flux;

/**
 * Outils de manipulation des Flux.
 * @author Adrien Colson
 */
public final class FluxUtils {

	/**
	 * Transforme un flux en flux qui émet la première valeur reçue dans une liste, puis émet périodiquement des listes des suivantes tant qu'il en reçoit, avant de recommencer.
	 * Sert principalement à ne pas faire de mises à jour trop fréquentes des UIs.
	 * @param <T> type de valeur émise par le flux
	 * @param source flux source
	 * @param duration durée de l'interval
	 * @return flux transformé
	 */
	public static <T> Flux<List<T>> intervalBuffer(final Flux<T> source, final Duration duration) {
		return source.publish(flux ->
			flux.take(1).collectList()
				.concatWith(flux.take(duration).collectList().filter(list -> !list.isEmpty())
					.repeatWhen(rf -> rf.takeWhile(nb -> nb > 0)))
				.repeatWhen(repeatFlux -> repeatFlux.takeUntilOther(flux.ignoreElements())));
	}

	/** Cache le constructeur, puisque c'est une classe utilitaire. */
	private FluxUtils() {
	}

}
