package fr.univlorraine.tools.vaadin.spring.security;

import java.util.Optional;

import org.springframework.security.core.context.SecurityContext;
import org.springframework.security.core.context.SecurityContextHolderStrategy;
import org.springframework.security.core.context.SecurityContextImpl;
import org.springframework.security.web.context.HttpSessionSecurityContextRepository;
import org.springframework.util.Assert;

import com.vaadin.server.VaadinSession;
import com.vaadin.ui.UI;

import lombok.extern.slf4j.Slf4j;

/**
 * A custom {@link SecurityContextHolderStrategy} that stores the {@link SecurityContext} in the Vaadin Session and in an InheritableThreadLocal CONTEXT_HOLDER.
 * @see <a href="https://github.com/peholmst/SpringSecurityDemo/blob/master/hybrid-security/src/main/java/org/vaadin/peholmst/samples/springsecurity/hybrid/VaadinSessionSecurityContextHolderStrategy.java">SpringSecurityDemo/.../VaadinSessionSecurityContextHolderStrategy.java</a>
 * @author Adrien Colson
 *
 * @see java.lang.ThreadLocal
 */
@Slf4j
public class VaadinSecurityContextHolderStrategy implements SecurityContextHolderStrategy {

	/** context_holder. */
	private static final ThreadLocal<SecurityContext> CONTEXT_HOLDER = new InheritableThreadLocal<SecurityContext>();

	/**
	 * @see org.springframework.security.core.context.SecurityContextHolderStrategy#clearContext()
	 */
	@Override
	public void clearContext() {
		CONTEXT_HOLDER.remove();
	}

	/**
	 * @see org.springframework.security.core.context.SecurityContextHolderStrategy#getContext()
	 */
	@Override
	public SecurityContext getContext() {
		SecurityContext ctx = (SecurityContext) Optional.ofNullable(UI.getCurrent())
			.map(UI::getSession)
			.map(VaadinSession::getSession)
			.map(httpSession -> {
				try {
					return httpSession.getAttribute(HttpSessionSecurityContextRepository.SPRING_SECURITY_CONTEXT_KEY);
				} catch (final IllegalStateException e) {
					return null;
				}
			})
			.orElse(CONTEXT_HOLDER.get());

		if (ctx == null) {
			ctx = createEmptyContext();
			setContext(ctx);
		}

		return ctx;
	}

	/**
	 * @see org.springframework.security.core.context.SecurityContextHolderStrategy#setContext(org.springframework.security.core.context.SecurityContext)
	 */
	@Override
	public void setContext(final SecurityContext context) {
		Assert.notNull(context, "Only non-null SecurityContext instances are permitted");

		CONTEXT_HOLDER.set(context);

		Optional.ofNullable(UI.getCurrent())
			.map(UI::getSession)
			.map(VaadinSession::getSession)
			.ifPresent(httpSession -> {
				try {
					httpSession.setAttribute(HttpSessionSecurityContextRepository.SPRING_SECURITY_CONTEXT_KEY, context);
				} catch (final IllegalStateException e) {
					log.trace("Can't set context attribute in vaadin session.");
				}
			});
	}

	/**
	 * @see org.springframework.security.core.context.SecurityContextHolderStrategy#createEmptyContext()
	 */
	@Override
	public SecurityContext createEmptyContext() {
		return new SecurityContextImpl();
	}

}
